import { state, dom, LOBBY_SCAN_THRESHOLD } from './state.js';
import * as ui from './ui.js';

export function getInitialData() {
    return window.electronAPI.getInitialData();
}

export function loadGamesFromConsole(consoleName) {
    state.selectedConsoleName = consoleName;
    state.isShowingFavorites = false;
    state.isShowingDownloaded = false;
    
    document.body.classList.toggle('arcade-active', consoleName?.startsWith('Arcade'));
    
    if (consoleName) {
        ui.handleNewGameList(state.allGames[consoleName] || []);
    } else {
        ui.handleNewGameList([]);
    }
}

export async function triggerPlayGame(gamePath) {
    if (state.isEmuAppRunning || !gamePath) return;
    await ui.initiateLaunchSequence(gamePath, 'standard');
}

export async function triggerPlayNetplayHost(gamePath) {
    if (state.isEmuAppRunning || !gamePath) return;
    await ui.initiateLaunchSequence(gamePath, 'netplay');
}

export async function triggerPlayNetplayClient(gameData) {
    if (state.isEmuAppRunning) return;
    if (!gameData || !gameData.game_name || !gameData.console_name || !gameData.ip || !gameData.port) return;

    const gameDataForBackend = { ...gameData };

    if (gameDataForBackend.console_name.startsWith('Arcade')) {
        const arcadeEntry = Object.entries(state.arcadeTitles).find(
            ([romName, prettyName]) => prettyName === gameDataForBackend.game_name
        );

        if (arcadeEntry) {
            const originalRomName = arcadeEntry[0];
            gameDataForBackend.game_name = originalRomName;
        } else {
            ui.showErrorInProgressBar(`Error: Could not find original ROM for ${gameDataForBackend.game_name}`);
            console.error("Could not perform reverse lookup for Arcade title:", gameDataForBackend.game_name);
            return;
        }
    }

    ui.prepareForOperation('play-netplay-client');
    dom.progressStatus.textContent = `Preparing to join ${gameData.game_name}...`;
    try {
        const pid = await window.electronAPI.playNetplayClient({
            gameName: gameDataForBackend.game_name,
            consoleName: gameDataForBackend.console_name,
            ip: gameDataForBackend.ip,
            port: gameDataForBackend.port,
            mitm_ip: gameDataForBackend.mitm_ip,
            mitm_port: gameDataForBackend.mitm_port,
            mitm_session: gameDataForBackend.mitm_session
        });
        if (pid) state.currentEmuAppPid = pid;
        else ui.showErrorInProgressBar('Internal Error: Invalid Process ID');
    } catch (error) {
        ui.showErrorInProgressBar(`Error: ${error.message || 'Failed to join game'}`);
    }
}

export async function triggerSaveGameApp(gamePath) {
    if (state.isEmuAppRunning || !gamePath) return;
    ui.prepareForOperation('save-app', gamePath);
    dom.progressStatus.textContent = 'Preparing to save game...';
    try {
        const pid = await window.electronAPI.saveGameApp(gamePath);
        state.currentEmuAppPid = pid || null;
    } catch (error) {
        ui.showErrorInProgressBar(`Error: ${error.message || 'Save failed'}`);
    }
}

export async function triggerSaveGameExternal(gamePath) {
    if (state.isEmuAppRunning || !gamePath) return;
    try {
        const gameBaseName = gamePath.split(/[/\\]/).pop().replace(/\.[^/.]+$/, "");
        const result = await window.electronAPI.showSaveDialog({
             title: 'Select Folder to Save Game',
             buttonLabel: 'Save Here',
             properties: ['openDirectory', 'createDirectory'],
             defaultPath: gameBaseName
        });

        if (!result.canceled && result.filePath) {
            ui.prepareForOperation('save-external', gamePath);
            dom.progressStatus.textContent = 'Preparing to save game externally...';
            const pid = await window.electronAPI.saveGameExternal({ gamePath, externalPath: result.filePath });
            state.currentEmuAppPid = pid || null;
        }
    } catch (error) {
         ui.showErrorInProgressBar(`Error: ${error.message || 'External save failed'}`);
    }
}

export async function showOnlineLobby() {
    if (state.isInitialSelectionActive || state.isEmuAppRunning) return;
    ui.closeTabMenu();
    ui.openLobbyMenu();
    dom.lobbyListUl.innerHTML = '<li>Searching for online games...</li>';
    try {
        const games = await window.electronAPI.getLobbyList();
        state.lobbyGames = games;
        ui.renderLobbyList();
        if (state.lobbyGames.length > 0) {
            state.lobbySelectionIndex = 0;
            ui.updateLobbySelection();
        }
    } catch (error) {
        state.lobbyGames = [];
        dom.lobbyListUl.innerHTML = '<li>Error fetching online games. Please try again later.</li>';
    }
}

export async function scanLobbyAndShowAlert() {
    try {
        const games = await window.electronAPI.getLobbyList();
        if (games && games.length >= LOBBY_SCAN_THRESHOLD) {
            ui.showNetplayAlert();
        } else {
            ui.hideNetplayAlert();
        }
    } catch (error) {
        console.warn("Silent lobby scan failed, hiding alert.", error);
        ui.hideNetplayAlert();
    }
}

export async function showDownloadedGamesList() {
    if (!state.selectedConsoleName) {
        ui.closeTabMenu();
        return;
    }
    const allConsoleGames = state.allGames[state.selectedConsoleName] || [];
    
    try {
        const downloadedGameNames = await window.electronAPI.getDownloadedGames(state.selectedConsoleName);
        const downloadedGames = allConsoleGames.filter(g => downloadedGameNames.includes(g.baseName));

        state.currentGames = downloadedGames;
        state.isShowingDownloaded = true;
        state.isShowingFavorites = false;
        ui.renderGameList(state.currentGames);
        ui.closeTabMenu();
        state.navigationMode = 'gameList';
        if (downloadedGames.length > 0) {
            ui.selectGame(0, true, false);
        } else {
            state.selectedIndex = -1;
            ui.updateGameImage(-1);
        }
    } catch (error) {
        console.error("Failed to get downloaded games list:", error);
        ui.renderGameList([]);
    }
}

export function killEmuApp() {
    if (state.currentEmuAppPid) {
        window.electronAPI.killEmuApp(state.currentEmuAppPid);
    }
}

export function savePromptResponse(decision) {
    const tempPath = state.gamePathToPotentiallySave;
    const platform = state.platformForPotentialSave;
    ui.hideSavePrompt();
    if (!tempPath || !platform) return;
    if (decision === 'yes') {
        ui.prepareForOperation('move-to-saved');
        dom.progressStatus.textContent = 'Saving game to collection...';
        dom.progressCancelHint.classList.add('hidden');
    }
    window.electronAPI.savePromptResponse({ decision, tempPath, platform });
}

export function toggleFavorite() {
    if (state.selectedIndex !== -1 && state.currentGames[state.selectedIndex]) {
        const game = state.currentGames[state.selectedIndex];
        window.electronAPI.toggleFavorite({
            gameBaseName: game.baseName,
            consoleFolderName: state.selectedConsoleName
        });
        
        const favList = state.favoriteGames[state.selectedConsoleName] || [];
        const isFav = favList.includes(game.baseName);
        if (isFav) {
            state.favoriteGames[state.selectedConsoleName] = favList.filter(f => f !== game.baseName);
        } else {
            favList.push(game.baseName);
            state.favoriteGames[state.selectedConsoleName] = favList;
        }
        ui.updateFavoriteStatusInList(game.baseName);
    }
}

export function manageSavedGames() {
    window.electronAPI.openSavedGamesFolder();
}

export function addToDownloadQueue() {
    if (state.currentGamePathForSaveMenu && state.selectedConsoleName) {
        const filename = state.currentGamePathForSaveMenu.split(/[/\\]/).pop();
        window.electronAPI.addToQueue({ consoleName: state.selectedConsoleName, filename });
        ui.closeSaveManageMenu();
    }
}

export function quitApp() {
    window.electronAPI.quitApp();
}

export function triggerSuspend() {
    window.electronAPI.suspendPc();
}

export function triggerShutdown() {
    window.electronAPI.shutdownPc();
}

export function triggerRestart() {
    window.electronAPI.restartPc();
}

export async function getUsername() {
    return await window.electronAPI.getUsername();
}

export function setUsername(username) {
    window.electronAPI.setUsername(username);
}

export async function fetchNotification() {
    const NOTIFICATION_URL = 'https://raw.githubusercontent.com/ex-ds-dev/CBnotification/refs/heads/main/CloudBoxNotification';
    try {
        const response = await fetch(NOTIFICATION_URL);
        if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
        const data = await response.json();
        state.notification.data = data;
    } catch (error) {
        console.error("Failed to fetch notification data:", error);
        state.notification.data = null;
    }
}

export async function getRelayConfig() {
    return await window.electronAPI.getRelayConfig();
}

export function setRelayConfig(config) {
    window.electronAPI.setRelayConfig(config);
}

export function setupApiListeners() {
    window.electronAPI.onEmuProgress(ui.handleProgressUpdate);
    window.electronAPI.onEmuExit(ui.handleEmuExit);
    window.electronAPI.onShowSavePrompt(({ tempPath, platform }) => {
        if (state.isEmuAppRunning || state.isSavePromptActive || state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isVersionMenuActive || state.isTabMenuActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
        ui.showSavePrompt(tempPath, platform);
    });
    window.electronAPI.onFolderOpenError(message => {
        console.error(message);
        alert(message);
    });
    window.electronAPI.onEmuLog(message => console.log(`[emu_app log]: ${message}`));
}